/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/***********************************************************************
* File name     : host.c
************************************************************************
* Created by    : Joerg Striegel, BID R&D, 05.06.96
* Last modified : 10.06.96
************************************************************************
* File contents :
* Download and upload functions between host/file and system/internal
* memory.
************************************************************************/

#if defined(_MSC_VER)
# pragma data_seg("host_c","FAR_DATA")
#elif defined(__BORLANDC__)
# pragma option -dc
# pragma warn -eff
#endif /* _MSC_VER etc. */


#ifdef BEST_FIRMWARE

#include <capitype.h>
#include <bios_malloc.h>

static void *malloc(b_int32 n)
{
  void *ptr;
  HWMalloc(&ptr, n);
  return ptr;
}


static void free(void *ptr)
{
  HWFree(ptr);
}


#else   /* #ifdef BEST_FIRMWARE */
#  define UPLOAD_RETRY_HACK

#include "lasterr.h"
#include <typedefs.h>
#include "b_cmd.h"

#endif  /* BEST_FIRMWARE */


#include <board.h>
#include <dynamic.h>
#include <errcapi.h>
#include <hostdefs.h>
#include <host.h>
#include <iocommon.h>
#include <master.h>
#include <regconst.h>
#include <regx20.h>
#include <regx50.h>
#include <session.h>

#ifndef BEST_FIRMWARE
#include <identos.h>
#endif

#define FAST_HOST_THRESHOLD 112

/* The define has to meet the following constraints:
   - < 127 for E2925
   - < 119 for padding
   - multiple of 8
   - as large as possible
   CZ
   */


/* *********************************************************************
 * host sysmem fill/dump: not in firmware
 * *********************************************************************/

#ifndef E2921A_MIN

#ifndef BEST_FIRMWARE

#define FOUR_X(a)      a, a, a, a
#define THIRTY_TWO_X(a) FOUR_X(FOUR_X(a)), FOUR_X(FOUR_X(a))

static b_int32 sm_command[MAXHANDLES] = {
  THIRTY_TWO_X(B_CMD_MEM_READ),
  THIRTY_TWO_X(B_CMD_MEM_READ),
  THIRTY_TWO_X(B_CMD_MEM_READ),
  THIRTY_TWO_X(B_CMD_MEM_READ)
};

static b_int32 sm_bufsize[MAXHANDLES] = {
  THIRTY_TWO_X(0),
  THIRTY_TWO_X(0),
  THIRTY_TWO_X(0),
  THIRTY_TWO_X(0)
};

static b_int32 sm_verify[MAXHANDLES] = {
  THIRTY_TWO_X(0),
  THIRTY_TWO_X(0),
  THIRTY_TWO_X(0),
  THIRTY_TWO_X(0)
};
#ifndef MIN
#define MIN(a,b) ( (a) < (b) ? (a) : (b) )
#endif

#define ADDR_INCREMENT64(bad, badhi, increment) {\
   if ( (bad) + (increment) < (bad) ) \
     { B_TRY_LICENSE(B_CAPABILITY_64_BIT); (badhi)++; } \
   (bad) += (increment); }


/* transfers any number of dwords using bytens given.
   IMPORTANT: buscommand must be setup in advance! */
static b_errtype DoTransfer(b_handletype handle,
    b_int32 busaddress,
    b_int32 busaddresshi,
    b_int32 intaddress,
    b_int32 byten,
    b_int32 nofdwords)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    b_int32 status;
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_BUSADDR, busaddress));
    if (!BestIs2925(handle))
    {
      /* all except 2925 */
      if (busaddresshi)
      {
        B_TRY(BestMasterBlockPropSet(handle, B_BLK_BUSADDR_HI, busaddresshi));
        B_TRY(BestMasterBlockPropSet(handle, B_BLK_BUSDAC, busaddresshi ? 1UL : 0UL));
      }
      B_TRY(BestMasterBlockPropSet(handle, B_BLK_COMPOFFS, intaddress));
    }

    B_TRY(BestMasterBlockPropSet(handle, B_BLK_BYTEN, byten));
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_INTADDR, intaddress));
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_NOFDWORDS, nofdwords));

    B_TRY(BestMasterBlockRun(handle));
    do
    {
      B_TRY(BestStatusRegGet(handle, &status));
      B_TRY_FAIL(status & 0x80 ? B_E_MASTER_ABORT : B_E_OK);
    } while (status & 1);
  }

  B_ERRETURN(B_TRY_RET);
}


/* transfer contigous block from internal memory to system memory or back.
   NOTE: buscommand etc. must be setup in advance! */
static b_errtype IntSysMemTransfer(b_handletype handle,
    b_int32 busaddress,
    b_int32 busaddresshi,
    b_int32 intaddress,
    b_int32 numbytes)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    b_int32 padding = busaddress % 4;
    b_int32 bytestransfered;
    b_int32 byten;
    if (padding)                /* leading, not dword-aligned bytes */
    {
      bytestransfered = MIN(4 - padding, numbytes);
      byten = 0x0FUL & ~(((1UL << (int) bytestransfered) - 1UL) << ((int) padding));
      B_TRY(DoTransfer(handle,
          busaddress & ~0x3UL, busaddresshi,
          intaddress & ~0x3UL,
          byten, 1UL));

      ADDR_INCREMENT64(busaddress, busaddresshi, bytestransfered);
      intaddress += bytestransfered;
      numbytes -= bytestransfered;
    }

    if (numbytes > 4)           /* full, dword-aligned blocks */
    {
      bytestransfered = 4 * (numbytes / 4);
      byten = 0x0000;
      B_TRY(DoTransfer(handle,
          busaddress & ~0x3, busaddresshi,
          intaddress & ~0x3,
          byten, numbytes / 4));

      ADDR_INCREMENT64(busaddress, busaddresshi, bytestransfered);
      intaddress += bytestransfered;
      numbytes -= bytestransfered;
    }

    if (numbytes > 0)           /* trailing, not completely filled blocks */
    {
      byten = 0x0FUL & ~((1UL << (int) numbytes) - 1UL);
      B_TRY(DoTransfer(handle,
          busaddress & ~0x3UL, busaddresshi,
          intaddress & ~0x3UL,
          byten, 1UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}

b_errtype EXPORT BestHostSysMemAccessPrepare(
    b_handletype handle,
    b_int32 buscmd,
    b_int32 bufsize
)
{
  B_DECLARE_FUNCNAME("BestHostSysMemAccessPrepare [hsmaprep]");

  B_TRY_VARS_NO_PROG;
  b_int32 DATAMEM_SIZE_IN_DWORDS;
  b_generic_infotype *mygen;
  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY(BestGenInfoGet(handle, B_PARAM_DATA_MEM, &mygen));
    DATAMEM_SIZE_IN_DWORDS = mygen->depth / 4;
    B_TRY_FCT_PARAM_RANGE(bufsize, 2, DATAMEM_SIZE_IN_DWORDS);
    /* prepare the master to do an immediate run .. */

    sm_verify[handle] = 0;
    if ((buscmd & B_HOST_VERIFY) == B_HOST_VERIFY)
    {
      if (BestIs2925(handle))
      {
        B_TRY_ERROR(B_E_NOT_E2925A);
      }
      else
      {
        sm_verify[handle] = 1;
      }
    }

    sm_bufsize[handle] = bufsize;
    sm_command[handle] = buscmd & 0xFF;

    B_TRY(BestMasterGenPropSet(handle, B_MGEN_RUNMODE,
        (b_int32) B_RUNMODE_IMMEDIATE));
    B_TRY(BestMasterGenPropSet(handle, B_MGEN_ATTRMODE,
        (b_int32) B_ATTRMODE_BLOCK));
    B_TRY(BestMasterGenPropSet(handle, B_MGEN_REPEATMODE,
        (b_int32) B_REPEATMODE_SINGLE));
    B_TRY(BestMasterGenPropSet(handle, B_MGEN_MASTERENABLE, 1UL));
    /* set the block properties */
    B_TRY(BestMasterBlockPropDefaultSet(handle));
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_ATTRPAGE, 0UL));
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_BUSCMD, sm_command[handle]));

    /* clear master aborts and compare errors */
    B_TRY(BestStatusRegClear(handle, (b_int32) (0x80 | 0x20)));
  }

  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestHostSysMemFill64(
    b_handletype handle,
    b_int32 bus_addr_low,
    b_int32 bus_addr_high,
    b_int32 num_of_bytes,
    b_int32 blocksize,
    b_int8ptr data_ptr
)
{
  B_DECLARE_FUNCNAME("BestHostSysMemFill64 [hsmfill64]");

  B_TRY_VARS_NO_PROG;

  b_int32 bufsize;              /* max size if int. buffer */
  b_int32 realblocksize;        /* actual size of bytes transfered */
  b_int32 realbufsize;          /* real size of int. buffer used */
  b_int32 intaddr;              /* internal address used */
  b_int32 startintaddr;         /* start of internal buffer */
  b_int32 alignoperand;         /* for alignment of internal address */
  b_int32 alignoffs = 0xFFFFFFFFUL; /* offset between int. buffer and address */
  b_generic_infotype *mygen;
  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    if (num_of_bytes)
    {
      /* parameter checking */
      B_TRY(BestGenInfoGet(handle, B_PARAM_DATA_MEM, &mygen));
      B_TRY_FCT_PARAM_R(2,
        BestIs2925(handle) ? (bus_addr_high ? 1 : 0) : 0,
        "E2925A cannot issue DAC"  /* no comma here */
        "bus_addr_high must be zero");

      if (bus_addr_high)
      {
        B_TRY_LICENSE(B_CAPABILITY_64_BIT);
      }

      B_TRY_FCT_PARAM_NULL_POINTER(data_ptr);
      bufsize = 4 * sm_bufsize[handle];
      B_TRY_FCT_PARAM_RANGE(blocksize, 1, bufsize);

      /* BestHostSysMemAccessPrepare() set a READ ? (we can only write) */
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_CMD_MEM_WRITE);
      BestLastErrorParamSet(handle, B_ERRPAR_2, sm_command[handle]);
      B_TRY_FAIL(sm_command[handle] == B_CMD_MEM_WRITE ?
        B_E_OK : B_E_SYSMEM_PREPCMD);

      /* proper (non)alignment for 25 or 26 */
      alignoperand = mygen->num_blocks;
      startintaddr = mygen->depth - bufsize;
      intaddr = mygen->depth;   /* force copying of data later */

      do
      {
        /* number of bytes to be transferred in one chunk */
        realblocksize = MIN(blocksize, num_of_bytes);

        if (intaddr + realblocksize > mygen->depth)
        {
          /* refill datamemory and set intaddr accordingly */

          /* take address part of startintaddr and alignment part of
           * bus_addr! */
          alignoffs = bus_addr_low & alignoperand;
          intaddr = (startintaddr & ~alignoperand) | alignoffs;

          if (intaddr < startintaddr) /* below bounds? */
            intaddr += (alignoperand + 1);  /* add qword/dword! */

          /* now compute size of internal buffer */
          realbufsize = MIN(mygen->depth - intaddr, num_of_bytes);
          if (realblocksize > realbufsize)
            realblocksize = realbufsize;

          /* n-times realblocksize or rest of data must fit into realbufsize */
          if (realbufsize < num_of_bytes)
            realbufsize -= (realbufsize % realblocksize);

          B_TRY(BestDataMemWrite(handle, intaddr,
              realbufsize, data_ptr));
          /* remove bytes really transferred from buffer */
          data_ptr += realbufsize;
        }

        assert(intaddr + realblocksize <= mygen->depth);

        B_TRY(IntSysMemTransfer(handle,
            bus_addr_low, bus_addr_high,
            intaddr,
            realblocksize));

        if (sm_verify[handle])  /* verify write access by reading back! */
        {
          b_int32 status;
          B_TRY(BestMasterBlockPropSet(handle, B_BLK_BUSCMD,
              (b_int32) B_CMD_MEM_READ));
          B_TRY(BestMasterBlockPropSet(handle, B_BLK_COMPFLAG, 1UL));

          B_TRY(IntSysMemTransfer(handle,
              bus_addr_low, bus_addr_high,
              intaddr,
              realblocksize));

          /* check for compare errors */
          B_TRY(BestStatusRegGet(handle, &status));
          B_TRY_FAIL(status & 0x20 ? B_E_SYSMEM_DATACOMPARE : B_E_OK);

          B_TRY(BestMasterBlockPropSet(handle, B_BLK_COMPFLAG, 0UL));
          B_TRY(BestMasterBlockPropSet(handle, B_BLK_BUSCMD,
              sm_command[handle]));
        }

        num_of_bytes -= realblocksize;  /* realblocksize never > # bytes! */
        intaddr += realblocksize;
        ADDR_INCREMENT64(bus_addr_low, bus_addr_high, realblocksize);

      } while (num_of_bytes > 0);
    }
  }

  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestHostSysMemFill(
    b_handletype handle,
    b_int32 bus_addr,
    b_int32 num_of_bytes,
    b_int32 blocksize,
    b_int8ptr data_ptr
)
{
  B_ERRETURN(BestHostSysMemFill64(handle, bus_addr, 0UL, num_of_bytes, blocksize, data_ptr));
}


b_errtype EXPORT BestHostSysMemDump(
    b_handletype handle,
    b_int32 bus_addr,
    b_int32 num_of_bytes,
    b_int32 blocksize,
    b_int8ptr data_ptr
)
{
  B_ERRETURN(BestHostSysMemDump64(handle, bus_addr, 0UL, num_of_bytes, blocksize, data_ptr));
}


b_errtype EXPORT BestHostSysMemDump64(
    b_handletype handle,
    b_int32 bus_addr_low,
    b_int32 bus_addr_high,
    b_int32 num_of_bytes,
    b_int32 blocksize,
    b_int8ptr data_ptr
)
{
  B_DECLARE_FUNCNAME("BestHostSysMemDump64 [hsmdump64]");

  B_TRY_VARS_NO_PROG;

  b_int32 bufsize;              /* copy of sm_bufsize for range checking */
  b_int32 realblocksize;
  b_int32 realbufsize;          /* bytes in int. buffer to be transfered */
  b_int32 intaddr;
  b_int32 startintaddr;         /* start of internal buffer */
  b_int32 alignoperand;         /* for alignment of internal address */
  b_int32 alignoffs = 0xFFFFFFFFUL; /* offset between int. buffer and address */
  b_generic_infotype *mygen;

  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    if (num_of_bytes)
    {
      B_TRY(BestGenInfoGet(handle, B_PARAM_DATA_MEM, &mygen));
      B_TRY_FCT_PARAM_R(2,
        BestIs2925(handle) ? (bus_addr_high ? 1 : 0) : 0,
        "E2925A cannot issue DAC, bus_addr_high must be zero");

      if (bus_addr_high)
      {
        B_TRY_LICENSE(B_CAPABILITY_64_BIT);
      }

      B_TRY_FCT_PARAM_NULL_POINTER(data_ptr);
      bufsize = 4 * sm_bufsize[handle];
      B_TRY_FCT_PARAM_RANGE(blocksize, 1, bufsize);

      /* BestHostSysMemAccessPrepare() set a WRITE ? (we can only read) */
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_CMD_MEM_READ);
      BestLastErrorParamSet(handle, B_ERRPAR_2, sm_command[handle]);
      B_TRY_FAIL(sm_command[handle] == B_CMD_MEM_READ ?
        B_E_OK : B_E_SYSMEM_PREPCMD);

      /* proper alignment for 25 or 26 */
      alignoperand = mygen->num_blocks;
      startintaddr = mygen->depth - bufsize;
      intaddr = mygen->depth;   /* force recalcutation in loop */
      realblocksize = MIN(blocksize, num_of_bytes);

      /* this loop does it all: - compute offset of internal memory
       * (alignment!) - transfer from system memory - copy data from internal
       * memory into buffer */
      do
      {
        if (intaddr + realblocksize > mygen->depth)
        {
          alignoffs = bus_addr_low & alignoperand;  /* save for later */
          intaddr = (startintaddr & ~alignoperand) | alignoffs;

          if (intaddr < startintaddr) /* below bounds? */
            intaddr += (alignoperand + 1);  /* add qword/dword! */

          /* now compute size of internal buffer */
          realbufsize = MIN(mygen->depth - intaddr, num_of_bytes);
          if (realblocksize > realbufsize)
            realblocksize = realbufsize;

          /* n-times realblocksize or rest of data must fit into realbufsize */
          if (realbufsize < num_of_bytes)
            realbufsize -= (realbufsize % realblocksize);
        }

        B_TRY(IntSysMemTransfer(handle,
            bus_addr_low, bus_addr_high,
            intaddr,
            realblocksize));

        num_of_bytes -= realblocksize;  /* realblocksize is never > num_bytes */
        intaddr += realblocksize;
        ADDR_INCREMENT64(bus_addr_low, bus_addr_high, realblocksize);
        realblocksize = MIN(blocksize, num_of_bytes);

        if ((intaddr + realblocksize > mygen->depth) ||
          (num_of_bytes == 0))
        {
          /* need to reload datamemory */
          B_TRY(BestDataMemRead(handle, intaddr - realbufsize,
              realbufsize,
              data_ptr));
          data_ptr += realbufsize;  /* bytes transfered */
        }
      } while (num_of_bytes > 0);
    }

  }

  B_ERRETURN(B_TRY_RET);
}
/* this function generates byte enables from a byte/wordsize and
   a busaddress */
static b_errtype BuildByten(b_sizetype wordsize,
    b_int32 busaddr,
    b_int8ptr byten)
{
  b_int8 offset = (b_int8) (busaddr & 0x03);
  if (wordsize == B_SIZE_BYTE)
  {
    *byten = (b_int8) (0x01 << (int) offset);
  }
  else if (wordsize == B_SIZE_WORD)
  {
    if ((busaddr & 0x00000001) != 0)
      return B_E_NO_WORD_BOUNDARY;
    *byten = (b_int8) (0x03 << (int) offset);
  }
  else if (wordsize == B_SIZE_DWORD)
  {
    if ((busaddr & 0x00000003) != 0)
      return B_E_NO_DWORD_BOUNDARY;
    *byten = 0xF;
  }
  else
  {
    *byten = 0xFF;
  }

  *byten = (b_int8) (~(*byten) & 0x0f);

  return B_E_OK;
}
b_errtype EXPORT BestHostPCIRegSet(
    b_handletype handle,
    b_addrspacetype addrspace,
    b_int32 bus_addr,
    b_int32 reg_value,
    b_sizetype wordsize
)
{
  B_DECLARE_FUNCNAME("BestHostPCIRegSet [hprgset]");

  B_TRY_VARS_NO_PROG;

  b_int8 byten;
  b_int32 dwval;
  b_int32 intaddr;
  b_int32 buscmd;
  b_int32 offset;
  b_generic_infotype *mygen;    /* for datamem size */
  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY_FCT_PARAM_RANGE(addrspace, B_ASP_CONFIG, B_ASP_MEM);
    B_TRY_FCT_PARAM_RANGE(wordsize, B_SIZE_BYTE, B_SIZE_DWORD);

    /* build dword */
    offset = bus_addr & 0x03;
    B_TRY(BuildByten(wordsize, bus_addr, &byten));

    if (wordsize == B_SIZE_BYTE)
    {
      B_TRY_FCT_PARAM(reg_value, reg_value > 0xFFUL);
      dwval = reg_value << (int) (offset << 3);
    }
    else if (wordsize == B_SIZE_WORD)
    {
      B_TRY_FCT_PARAM(reg_value, reg_value > 0xFFFFUL);
      dwval = reg_value << (int) (offset << 3);
    }
    else
    {
      dwval = reg_value;
    }

    B_TRY(BestGenInfoGet(handle, B_PARAM_DATA_MEM, &mygen));
    intaddr = mygen->depth - (mygen->num_blocks + 1);
    intaddr |= (bus_addr & mygen->num_blocks);  /* proper alignment... */

    /* differentiate between address spaces */
    switch (addrspace)
    {

    case B_ASP_CONFIG:
    case B_ASP_CONFIG_TYPE1:
      /* CONFIG address space */
      bus_addr &= ~3;
      /* check, if we have to set bit 0 of busaddr for config cycle type 1 */
      if (addrspace == B_ASP_CONFIG_TYPE1)
        bus_addr++;
      buscmd = B_CMD_CONFIG_WRITE;
      break;

    case B_ASP_IO:
      /* IO address space */
      /* this now sets intaddr, cmd, apage, runmode etc. */
      buscmd = B_CMD_IO_WRITE;
      break;

    case B_ASP_MEM:
      /* MEM address space */
      bus_addr &= 0xfffffffcUL;
      buscmd = B_CMD_MEM_WRITE;
      break;

    default:
      assert(0 && "BestHostPCIRegSet: We should never be here");
    }

    B_TRY(BestHostSysMemAccessPrepare(handle, buscmd, 2UL));

    B_TRY(BestDataMemWrite(handle, intaddr & ~3, 4UL, (b_int8ptr) & dwval));
    B_TRY(DoTransfer(handle, bus_addr, 0UL, intaddr & ~3UL, (b_int32) byten, 1UL));
  }

  B_ERRETURN(B_TRY_RET);
}





b_errtype EXPORT BestHostPCIRegGet(
    b_handletype handle,
    b_addrspacetype addrspace,
    b_int32 bus_addr,
    b_int32 * regvalue_ptr,
    b_sizetype wordsize
)
{
  B_DECLARE_FUNCNAME("BestHostPCIRegGet [hprgget]");

  B_TRY_VARS_NO_PROG;

  b_int8 byten;
  b_int32 dwval;
  b_int32 intaddr;
  b_int32 buscmd;
  b_generic_infotype *mygen;    /* for datamem depth */
  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY_FCT_PARAM_RANGE(addrspace, B_ASP_CONFIG, B_ASP_MEM);
    B_TRY_FCT_PARAM_NULL_POINTER(regvalue_ptr);
    B_TRY_FCT_PARAM_RANGE(wordsize, B_SIZE_BYTE, B_SIZE_DWORD);

    B_TRY(BuildByten(wordsize, bus_addr, &byten));

    B_TRY(BestGenInfoGet(handle, B_PARAM_DATA_MEM, &mygen));
    intaddr = (mygen->depth - (mygen->num_blocks + 1));
    intaddr |= (bus_addr & mygen->num_blocks);  /* same align as busaddr */

    /* differentiate between address spaces */
    switch (addrspace)
    {
    case B_ASP_CONFIG:
    case B_ASP_CONFIG_TYPE1:
      /* CONFIG address space */
      bus_addr &= 0xfffffffcUL;
      /* check, if we have to set bit 0 of busaddr for config cycle type 1 */
      if (addrspace == B_ASP_CONFIG_TYPE1)
        bus_addr++;
      /* this now sets cmd, apage, runmode etc. */
      buscmd = B_CMD_CONFIG_READ;
      break;

    case B_ASP_IO:
      /* IO address space */
      buscmd = B_CMD_IO_READ;
      break;

    case B_ASP_MEM:
      /* MEM address space */
      /* now set busaddr, byten, cmd */
      bus_addr &= 0xfffffffcUL;
      /* this now sets intaddr, cmd, apage, runmode etc. */
      buscmd = B_CMD_MEM_READ;
      break;

    default:
      assert(0 && "BestHostPCIRegSet: We should never be here");
    }

    B_TRY(BestHostSysMemAccessPrepare(handle, buscmd, 2UL));
    B_TRY(DoTransfer(handle, bus_addr, 0UL, intaddr & ~3UL, (b_int32) byten, 1UL));

    /* get the dword from internal memory */
    B_TRY(BestDataMemRead(handle, intaddr, (b_int32) wordsize, (b_int8ptr) & dwval));

    /* get correct value from dword */
    switch (wordsize)
    {
    case B_SIZE_BYTE:
      *regvalue_ptr = dwval & 0x000000ff;
      break;

    case B_SIZE_WORD:
      *regvalue_ptr = dwval & 0x0000ffff;
      break;

    default:
      *regvalue_ptr = dwval;
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */

#endif  /* E2921A_MIN */

/* ********************************************************************
 * end of host functions
 * ********************************************************************/


#ifndef BEST_FIRMWARE

b_errtype EXPORT BestDataMemInit(b_handletype handle)
{
  B_TRY_VARS_NO_PROG;
  b_generic_infotype *GenericDataMemInfo;
  b_int32 NumBytes;

  B_TRY_BEGIN
  {
    B_TRY(BestGenInfoGet(handle, B_PARAM_DATA_MEM, &GenericDataMemInfo));
    NumBytes = GenericDataMemInfo->depth;
    B_TRY(BestDataMemPatternFill(handle, 0UL, NumBytes, 0UL, 0UL, B_PATT_CONST));
  }

  B_ERRETURN(B_TRY_RET);
}
#endif


#ifndef BEST_FIRMWARE

b_errtype EXPORT BestHostIntMemFill(
    b_handletype handle,
    b_int32 int_addr,           /* byte boundary */
    b_int32 num_of_bytes,
    b_int8ptr data_ptr
)
{
  B_ERRETURN(BestDataMemWrite(handle, int_addr, num_of_bytes, data_ptr));
}
#endif


b_errtype EXPORT BestDataMemWrite(
    b_handletype handle,
    b_int32 int_addr,           /* dword boundary */
    b_int32 num_of_bytes,
    b_int8ptr data_ptr
)
{
  B_DECLARE_FUNCNAME("BestDataMemWrite [datamwr]");

  B_TRY_VARS_NO_PROG;
  char reason[256];
  b_generic_infotype *mygen;
  b_int32 eff_int_addr, eff_num_of_bytes;
  b_int32 count;
  b_int8ptr data_buf = NULL;
  b_int8ptr real_buf = NULL;
  b_int32ptr dest_ptr = NULL;
  b_int8ptr src_ptr = NULL;

  B_TRY_BEGIN
  {
    b_porttype port;
    B_TRY_FAIL(num_of_bytes ? B_E_OK : B_E_INTERNAL_RETURN);

#ifndef BEST_FIRMWARE

    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY(BestGenInfoGet(handle, B_PARAM_DATA_MEM, &mygen));

    sprintf(reason, "greater than data memory (%ld)", mygen->depth);
    B_TRY_FCT_PARAM_R(1, int_addr > mygen->depth, reason);

    sprintf(reason, "exceeds data memory (%ld)", mygen->depth);
    B_TRY_FCT_PARAM_R(2, int_addr + num_of_bytes > mygen->depth, reason);

    B_TRY(BestGetPortFromHandle(handle, &port));

#endif

    if (num_of_bytes > FAST_HOST_THRESHOLD && Best16BitRegisterFile(handle))
    {

#ifndef BEST_FIRMWARE
      /* Block mode */
      switch (port)
      {
        /* accesses to data memory require the ichiban to be in init mode
         * which also means that during this time the target is in retry mode
         * -> no PCI data transfer in block mode possible, CZ */
      case B_PORT_PCI_CONF:
      case B_PORT_PCI_IO:

#ifdef WIN95
        /* Serial seems to have problems with the block mode. Don't allow it.
        /* CZ */
		/* Same for FASTHIF (CS) */
        /* CONSIDER: capi property */
        case B_PORT_RS232:
        case B_PORT_FASTHIF:

#endif

        /* do nothing...leave real_buf == NULL */
        break;

      default:
        /* default to low mem solution for 2925 */
        real_buf = (Best16BitRegisterFile(handle) ?
          malloc((size_t) (num_of_bytes + 16 + 2 * sizeof(b_int32))) :
          NULL);
        break;
      }                         /*lint !e788 ... not all enums used */

#endif

      if (real_buf)
        data_buf = real_buf + 2 * sizeof(b_int32);

      else
      {
        /* low memory solution */
        /* Note: variables have different meanings in this recursion */
        eff_int_addr = int_addr;
        data_buf = data_ptr;

        while (num_of_bytes)
        {
          eff_num_of_bytes = num_of_bytes > FAST_HOST_THRESHOLD ?
            FAST_HOST_THRESHOLD : num_of_bytes;
          B_TRY (BestDataMemWrite(handle, eff_int_addr,
				  eff_num_of_bytes, data_ptr));
          eff_int_addr += eff_num_of_bytes;
          num_of_bytes -= eff_num_of_bytes;
          data_ptr += eff_num_of_bytes;
        }

        B_TRY_FAIL (B_E_INTERNAL_RETURN);
        /* low memory solution done */
      }
    }                           /* block mode */

    else
    {
      real_buf = malloc((size_t) (num_of_bytes + 16 + 2 * sizeof(b_int32)));
      B_TRY_FAIL (real_buf ? B_E_OK : B_E_HOST_MEM_FULL);

      /* Note: This is a very special trick. For the E2926A we need 2 long
       * variables as parameters at the beginning of the buffer. That is why
       * the 2 * sizeof (b_int32) is added to the required size for malloc.
       * Now we increment the pointer for the padding work and decrement it
       * back when we need the additional 2 longs. CZ */

      data_buf = real_buf + 2 * sizeof(b_int32);
    }

    /* at this point we have a qword aligned buffer */
    eff_int_addr = int_addr & 0xfffffff8UL;

    for (eff_num_of_bytes = num_of_bytes & 0xfffffff8UL;
      eff_int_addr + eff_num_of_bytes < int_addr + num_of_bytes;)
      eff_num_of_bytes += 8;

    /* now we have all qword aligned effective addresses, get pad bytes now */
    if (eff_int_addr != int_addr)
    {
      B_TRY(BestDataMemRead(handle, eff_int_addr, 8UL, data_buf));
    }

    if ((int_addr + num_of_bytes) & 0x00000007)
    {
      B_TRY(BestDataMemRead(handle, eff_int_addr + eff_num_of_bytes - 8,
          8UL, &data_buf[eff_num_of_bytes - 8]));
    }

    /* copy source into aligned buffer */
    memcpy(&data_buf[int_addr - eff_int_addr], data_ptr, (size_t) num_of_bytes);

    /* now effectively write the data to the card */
    if (Best16BitRegisterFile(handle))  /* new protocol and new hw (E2926A) */
    {
      b_int8ptr ptr;
      b_int8ptr p;            /* little helpers */

      count = eff_num_of_bytes / sizeof(b_int32);
      dest_ptr = (b_int32ptr) data_buf;
      src_ptr = data_buf;

      while (count--)
      {
        *dest_ptr = ((b_int32) src_ptr[0] << 24) |
          ((b_int32) src_ptr[1] << 16) |
          ((b_int32) src_ptr[2] << 8) |
          (b_int32) src_ptr[3];
        dest_ptr++;
        src_ptr += 4;
      }

#ifndef BEST_FIRMWARE
      if (eff_num_of_bytes <= FAST_HOST_THRESHOLD + 8) {

#endif

        ptr = real_buf;
        p = BestLong2Stream(ptr, &eff_int_addr, 1UL);
        p = BestLong2Stream(p, &eff_num_of_bytes, 1UL);
        B_TRY(BestBasicCommandVariable(handle, CMD_EXEFHIF_HOSTINTMEMFILL,
            ptr, (b_int16) (eff_num_of_bytes + 8),
            NULL, NULL));

#ifndef BEST_FIRMWARE
      }

      else {
        B_TRY(BestBasicBlockCommand(handle, B_BIO_DATA_MEM,
            eff_int_addr, eff_num_of_bytes,
            8UL, CMD_WRITE, data_buf));
      }

#endif

    }

#ifndef BEST_FIRMWARE
    else {
      b_int8 byten = 0;
      B_TRY(BestBasicBlockWrite(handle, RAM_OFFS,
          (b_int8ptr) & eff_int_addr, 4UL));
      /* reset byte enable */
      B_TRY(BestBasicBlockWrite(handle, RAM_BYTEN, &byten, 1UL));
      /* using block-transfer */
      B_TRY(BestBasicBlockWrite(handle, RAM_DATA,
          (b_int8ptr) data_buf, eff_num_of_bytes));
    }

#endif

  }

  /* this traps the recursive situation where num_of_bytes == 0 */
  if (B_TRY_RET == B_E_INTERNAL_RETURN)
    B_TRY_RET = B_E_OK;

  /* did we malloc ? */
  if (real_buf)
    free(real_buf);

  B_ERRETURN(B_TRY_RET);
}


#ifndef BEST_FIRMWARE

b_errtype EXPORT BestHostIntMemDump(
    b_handletype handle,
    b_int32 int_addr,           /* byte boundary */
    b_int32 num_of_bytes,
    b_int8ptr data_ptr
)
{
  B_ERRETURN(BestDataMemRead(handle, int_addr, num_of_bytes, data_ptr));
}
#endif


b_errtype EXPORT BestDataMemRead(
    b_handletype handle,
    b_int32 int_addr,           /* byte boundary */
    b_int32 num_of_bytes,
    b_int8ptr data_ptr
)
{
  B_DECLARE_FUNCNAME("BestDataMemRead [datamrd]");

  B_TRY_VARS_NO_PROG;
  char reason[256];
  b_generic_infotype *mygen;
  b_int32 eff_int_addr, eff_num_of_bytes;
  b_int32 count;
  b_int8ptr data_buf = NULL;
  b_int32ptr dest_ptr = NULL;
  b_int8ptr src_ptr = NULL;

  B_FCT_PARAM_NULL_POINTER_CHK(data_ptr);

  B_TRY_BEGIN {
    b_porttype port;
    B_TRY_FAIL(num_of_bytes ? B_E_OK : B_E_INTERNAL_RETURN);

#ifndef BEST_FIRMWARE
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY(BestGenInfoGet(handle, B_PARAM_DATA_MEM, &mygen));

    sprintf(reason, "greater than data memory (%ld)", mygen->depth);
    B_TRY_FCT_PARAM_R(1, int_addr > mygen->depth, reason);

    sprintf(reason, "exceeds data memory (%ld)", mygen->depth);
    B_TRY_FCT_PARAM_R(2, int_addr + num_of_bytes > mygen->depth, reason);

    B_TRY(BestGetPortFromHandle(handle, &port));

#endif

    if (num_of_bytes > FAST_HOST_THRESHOLD && Best16BitRegisterFile(handle))
    {

#ifndef BEST_FIRMWARE
      switch (port)
      {
        /* accesses to data memory require the ichiban to be in init mode
         * which also means that during this time the target is in retry mode
         * -> no PCI data transfer in block mode possible, CZ */
      case B_PORT_PCI_CONF:
      case B_PORT_PCI_IO:

#ifdef WIN95
        /* Win95 seems to have problems with the block mode. Don't allow it.
         * CZ */
		/* Same for FASTHIF (CS) */
        /* CONSIDER: CAPI property ? */
      case B_PORT_RS232:
      case B_PORT_FASTHIF:

#endif  /* WIN95 */

        /* do nothing...leave data_buf == NULL */
        break;

      default:
        if ((GetOsVersionShort() == OS_WIN95) && ((port == B_PORT_RS232) || (port == B_PORT_FASTHIF))) {
            break;
        }
#ifndef _DOS  /* SCR; guaranteed problems with 32-bit mallocs! */

        /* default to low mem solution for 2925 (malloc for 2926) */
        if (Best16BitRegisterFile(handle))
        {
          data_buf = malloc((size_t) (num_of_bytes + 16));
          B_TRY_FAIL(data_buf ? B_E_OK : B_E_HOST_MEM_FULL);
        }
#endif  /* _DOS */

        break;
      }                         /*lint !e788 ... not all enums used */

#endif  /* BEST_FIRMWARE */

      if (NULL == data_buf)
      {
        /* low memory (or PCI) solution */
        /* Note: variables have different meanings in this recursion */
        eff_int_addr = int_addr;
        data_buf = data_ptr;

        /* we start a recursion here */
        while (num_of_bytes)
        {
          eff_num_of_bytes = 
            num_of_bytes > FAST_HOST_THRESHOLD ? 
            FAST_HOST_THRESHOLD : num_of_bytes;
          B_TRY (BestDataMemRead(handle, eff_int_addr,
				 eff_num_of_bytes, data_ptr));
          eff_int_addr += eff_num_of_bytes;
          num_of_bytes -= eff_num_of_bytes;
          data_ptr += eff_num_of_bytes;
        }

        data_buf = NULL; /*make sure data_buf is reset so we do not delete it*/
        B_TRY_FAIL (B_E_INTERNAL_RETURN);
        /* low memory solution done */
      }
    }                           /* blockmode */

    else
    {
      /* Hi Stu, whatever Lint said before here, it did not work */
      data_buf = malloc((size_t) (num_of_bytes + 16));
      B_TRY_FAIL (data_buf ? B_E_OK : B_E_HOST_MEM_FULL);
    }

    /* at this point we have a qword aligned buffer */
    eff_int_addr = int_addr & 0xfffffff8UL;

    /* the following loop will be taken 2 times max */
    for (eff_num_of_bytes = num_of_bytes & 0xfffffff8UL;
      eff_int_addr + eff_num_of_bytes < int_addr + num_of_bytes;)
      eff_num_of_bytes += 8;

    /* get data from card */
    if (Best16BitRegisterFile(handle))  /* new protocol and new hw (E2926A) */
    {
      b_errtype err = B_E_OK;
      b_int32 bytesread = 0;
      b_int32 blocksize = eff_num_of_bytes;
      b_int32 startaddr = eff_int_addr;
      b_int8ptr dataptr = data_buf;
#ifndef BEST_FIRMWARE
#  ifdef UPLOAD_RETRY_HACK
      int retrycounter = 0;
      while (bytesread < eff_num_of_bytes)
      {
#  endif /* defined upoad_retry_hack */
      
	if (eff_num_of_bytes <= FAST_HOST_THRESHOLD + 8)
	{
#endif /* not defined firmware */
	  b_int8 param[IN_EXEFHIF_HOSTINTMEMDUMP];
	  b_int8ptr ptr;          /* a little helper */
	  b_int16 outsize = OUT_EXEFHIF_HOSTINTMEMDUMP;
	  ptr = BestLong2Stream((b_int8ptr) param, &startaddr, 1UL);
	  ptr = BestLong2Stream(ptr, &blocksize, 1UL);
	  err = BestBasicCommand(handle, CMD_EXEFHIF_HOSTINTMEMDUMP,
				 param, IN_EXEFHIF_HOSTINTMEMDUMP,
				 dataptr, &outsize);
#ifndef BEST_FIRMWARE
	}
	else
	{
	  err = BestBasicBlockCommand(handle, B_BIO_DATA_MEM,
				      startaddr, blocksize,
				      8UL, CMD_READ, dataptr);
	}
#  ifdef UPLOAD_RETRY_HACK
	/* error code is B_E_ERROR in release build, only when first
	   error parameter set to B_EERR_PARTIAL_RCVE.
	   
	   in debug case, error code is B_E_PARTIAL_RCVE */
	
	if ( (err == B_E_ERROR || err == B_E_PARTIAL_RCVE) &&
	     retrycounter++ < 20 )
	{
	  b_int64 actErrorCode, numBytes, errpar3, errpar4, errpar5;
	  BestLastErrorAllParamsGet (handle,
				     &actErrorCode, &numBytes,
				     &errpar3, &errpar4, &errpar5);
	  
	  if (err == B_E_ERROR && actErrorCode != B_EERR_PARTIAL_RCVE)
	  {
	    B_ERRETURN (err);	/* not our retry-case! */
	  }

	  /* set read bytes to zero; retry.
	     Every 3rd time decrease buffer size by two */
	  bytesread = 0;
	  if (retrycounter % 2 == 0)
	    blocksize /= 2;
	  
	  continue;		/* do it again */
	}
	else if (err == B_E_OK)
	{
	  retrycounter = 0;	/* we made it */
	  bytesread += blocksize;
	  startaddr += blocksize; /* increase startaddr */
	  dataptr += blocksize;
	}
	else
	{
	  break;		/* get out of loop */
	}
      }
#  endif /* defined upload_retry_hack */
#endif /* not defined firmware */
      B_TRY_FAIL (err);

      count = eff_num_of_bytes / sizeof(b_int32);
      dest_ptr = (b_int32ptr) data_buf;
      src_ptr = data_buf;

      while (count--)
      {
        *dest_ptr = ((b_int32) src_ptr[0] << 24) |
          ((b_int32) src_ptr[1] << 16) |
          ((b_int32) src_ptr[2] << 8) |
          (b_int32) src_ptr[3];
        dest_ptr++;
        src_ptr += 4;
      }
    }

#ifndef BEST_FIRMWARE
    else {
      b_int8 byten = 0;
      B_TRY(BestBasicBlockWrite(handle, RAM_OFFS,
          (b_int8ptr) & eff_int_addr, 4UL));
      /* reset byte enable */
      B_TRY(BestBasicBlockWrite(handle, RAM_BYTEN, &byten, 1UL));
      /* using block-transfer */
      B_TRY(BestBasicBlockRead(handle, RAM_DATA,
          (b_int8ptr) data_buf, eff_num_of_bytes));
    }

#endif

    /* copy the data into user buffer */
    memcpy(data_ptr, &data_buf[int_addr - eff_int_addr], (size_t) num_of_bytes);
  }

  /* this traps the recursive situation where num_of_bytes == 0 */
  if (B_TRY_RET == B_E_INTERNAL_RETURN)
    B_TRY_RET = B_E_OK;

  /* did we malloc ? */
  if (data_buf)
    free(data_buf);

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * Fills data memory with a caller-defined pattern.  The meaning of
 * params value1 and value2 are dependent on the pattern as follows;
 * pattern          value1              value2
 * ==============   =================== =========================
 * B_PATT_CONST     mem set to this     ignored
 * B_PATT_TOGGLE    1st var             2nd var
 * B_PATT_COUNTER   starting val.       0==count down, else count up
 * B_PATT_WALKBIT0  ignored             ignored
 * B_PATT_WALKBIT1  ignored             ignored
 * B_PATT_RANDOM    seed                ignored
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestDataMemPatternFill(
    b_handletype handle,
    b_int32 int_addr,           /* MUST be quadword aligned */
    b_int32 num_of_bytes,       /* same */
    b_int32 value1,             /* meaning depends on pattern */
    b_int32 value2,             /* same */
    b_dmempatttype pattern          /* type of pattern to use */
)
{
  B_DECLARE_FUNCNAME("BestDataMemPatternFill [datampfill]");

  B_TRY_VARS_NO_PROG;

  size_t i;
  size_t *pDwordBuf;
  b_int32 NumDWords = num_of_bytes / 4;
  size_t check_size = (size_t)NumDWords;
  b_int32 dwPattern = (b_int32)pattern;
  B_FCT_PARAM_ALIGNMENT_CHK(int_addr, 8);
  B_FCT_PARAM_ALIGNMENT_CHK(num_of_bytes, 8);

  /* In an Ichiban this is all done onboard */
  if(BestHasIchiban(handle))
  {
    b_int8 params[IN_DATA_MEM_PATT_FILL];
    b_int8ptr ptr;          /* a little helper */
    ptr = BestLong2Stream((b_int8ptr) params, &int_addr, 1UL);
    ptr = BestLong2Stream(ptr, &num_of_bytes, 1UL);
    ptr = BestLong2Stream(ptr, &value1, 1UL);
    ptr = BestLong2Stream(ptr, &value2, 1UL);
    (void)BestLong2Stream(ptr, &dwPattern, 1UL);

    B_ERRETURN(BestBasicCommand(handle, CMD_DATA_MEM_PATT_FILL,
        params, IN_DATA_MEM_PATT_FILL, NULL, NULL));
  }

#ifndef BEST_FIRMWARE
  /* From here on we're only talking to a 2925 */

  /* check sizes ... 16-bit platforms */
  if(NumDWords > (b_int32)check_size)
  {
    B_FCT_PARAM_ERROR(3, "Exceeds 16-bit capability");
  }
  
  /* NOTE; as we'll be calling BestDataMemWrite() anyway we'll let it check
   * the data mem. bounds. */

  /* alloc memory */
  if (NULL == (pDwordBuf = (size_t *) malloc((size_t)num_of_bytes)))
    B_ERRETURN(B_E_HOST_MEM_FULL);

  B_TRY_BEGIN
  {
    switch (pattern)
    {
    case B_PATT_CONST:          /* value1 == const */
      for (i = 0; i < NumDWords; i++)
      {
        pDwordBuf[i] = value1;
      }
      break;

    case B_PATT_TOGGLE:         /* value1 == const1, value2 == const2 */
      for (i = 0; i < NumDWords; i += 2)
      {
        pDwordBuf[i] = value1;
        pDwordBuf[i + 1] = value2;
      }
      break;

    case B_PATT_COUNTER:        /* value1 == start, value2 == direction */
      if (value2)
      {
        /* going up */
        for (i = 0; i < NumDWords; i++)
        {
          pDwordBuf[i] = value1 + i;
        }
      }
      else
      {
        /* going down */
        for (i = 0; i < NumDWords; i++)
        {
          pDwordBuf[i] = value1 - i;
        }
      }
      break;

    case B_PATT_WALKBIT1:
      for (i = 0; i < NumDWords; i++)
      {
        pDwordBuf[i] = ((b_int32) 1 << (i % 32));
      }
      break;

    case B_PATT_WALKBIT0:
      for (i = 0; i < NumDWords; i++)
      {
        pDwordBuf[i] = ~((b_int32) 1 << (i % 32));
      }
      break;

    case B_PATT_RANDOM:         /* value1 == seed */
      srand((unsigned int)value1);
      for (i = 0; i < NumDWords; i++)
      {
        /* note; RAND_MAX is usually 0x7fff */
        pDwordBuf[i] = (b_int32) (((b_int16) rand() << 16) & (b_int16) rand());
      }
      break;

    default:
      B_TRY_FCT_PARAM_X(pattern, "Unknown pattern")
    }

    /* send to the Best */
    B_TRY(BestDataMemWrite(handle, int_addr, num_of_bytes, (b_int8ptr) pDwordBuf));

  }

  /* did we malloc ? */
  if (pDwordBuf)
    free(pDwordBuf);

  B_ERRETURN(B_TRY_RET);

#endif

}
